/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc.groups;

import rdc.RDCSample;

import com.inet.report.Area;
import com.inet.report.BorderProperties;
import com.inet.report.DatabaseField;
import com.inet.report.DatabaseTables;
import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.Group;
import com.inet.report.BySummaryOrder;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;

/**
 * This sample shows how to build a report with groups and sort data by summary order. The data will be grouped by
 * country and by position. The country-group will be sorted alphabetically, the position-group will be sorted by sum of
 * own salary values. This shows only least two groups, all other data will be grouped in "NewOtherName".<BR>
 * <BR>
 * The Data Source Configuration "Sample Database" is necessary for this sample. You can create it using the Data Source
 * Manager in i-net Designer or Configuration Manager.
 */
public class BySummaryOrderSample extends RDCSample {
    static final String[] EMPLOYEE_ADRESSES_NAMES = { "EmployeeID", "Country" };
    static final int[]    EMPLOYEE_ADRESSES_TYPES = { Field.NUMBER, Field.STRING };
    static final String[] EMPLOYEE_NAMES          = { "EmployeeID", "Position", "Salary" };
    static final int[]    EMPLOYEE_TYPES          = { Field.NUMBER, Field.STRING, Field.NUMBER };

    /**
     * Creates a new report, add database fields, groups and sort data by summary order.
     * @param exportFmt output format
     * @return engine the engine for the specified report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            // Set database connection
            Datasource ds = eng.getDatabaseTables().getDatasource( 0 ); // the default Datasource(the first) always exists. Otherwise call first createDatasource()
            ds.setDataSourceConfigurationName( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Fill the engine here            
            DatabaseTables sqlSources = eng.getDatabaseTables();

            // Create the tables
            TableSource ts_employeeAdrsses = ds.createTableSource( "EmployeeAddresses", "Employee_Addresses" );
            for( int colIdx = 0; colIdx < EMPLOYEE_ADRESSES_NAMES.length; colIdx++ ) {
                ts_employeeAdrsses.addColumn( EMPLOYEE_ADRESSES_NAMES[colIdx], EMPLOYEE_ADRESSES_TYPES[colIdx] );
            }

            TableSource ts_employee = ds.createTableSource( "Employees", "Employees" );
            for( int colIdx = 0; colIdx < EMPLOYEE_NAMES.length; colIdx++ ) {
                ts_employee.addColumn( EMPLOYEE_NAMES[colIdx], EMPLOYEE_TYPES[colIdx] );
            }

            // Join both tables
            sqlSources.addJoin( "Employee_Addresses", "EmployeeID", "Employees", "EmployeeID", DatabaseTables.JOINTYPE_INNER,
                                DatabaseTables.JOINLINK_EQUALS );

            // Add database fields to report
            Fields fields = eng.getFields();
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );

            DatabaseField country = fields.getDatabaseField( "Employee_Addresses.Country" );
            dSec.addFieldElement( country, 2000, 0, 2000, 300 );
            DatabaseField position = fields.getDatabaseField( "Employees.Position" );
            dSec.addFieldElement( position, 3000, 0, 2000, 300 );
            DatabaseField salary = fields.getDatabaseField( "Employees.Salary" );
            dSec.addFieldElement( salary, 4000, 0, 2000, 300 );

            // Set Record Selection Formula
            eng.setSF( "{Employees.Salary} >= 30000" );

            // Add a group
            Group groupCountry = eng.addGroup( country );

            // Add a groupname field to group header
            Area gh0Area = groupCountry.getHeader();
            Section gh0Sec = gh0Area.getSection( 0 );
            FieldElement gNField0a = gh0Sec.addFieldElement( fields.getGroupNameField( 0 ), 0, 0, 5425, 230 );
            gNField0a.setBold( true );

            // Add a second group
            Group groupPosition = eng.addGroup( position );

            // Add a groupname field to group header
            Area gh1Area = groupPosition.getHeader();
            Section gh1Sec = gh1Area.getSection( 0 );
            FieldElement gNField1a = gh1Sec.addFieldElement( fields.getGroupNameField( 1 ), 200, 0, 5425, 230 );
            gNField1a.setItalic( true );

            // Add some summary fields to group footer
            Area gf1Area = groupPosition.getFooter();
            Section gf1Sec = gf1Area.getSection( 0 );
            SummaryField smFieldGr1 = fields.addSummaryField( salary, SummaryField.SUM, "Sum of " + salary.getName() + "1" );
            smFieldGr1.setGroup( groupPosition );
            FieldElement summe = gf1Sec.addFieldElement( smFieldGr1, 6768, 0, 1043, 240 );
            summe.setBold( true );
            summe.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            summe.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Create BySummaryOrder with SummaryField and sorting direction
            BySummaryOrder groupSort = new BySummaryOrder();
            groupPosition.setSort( Group.SORT_SUMMARY_ORDER );
            groupPosition.setBySummaryOrder( groupSort );
            groupSort.addSummaryField( smFieldGr1 );

            // Set All Sorts - shows all values sorted ascending or descending
            //@see BySummaryOrder.GROUP_SORT_ALL
            /*groupSort.setSortType(BySummaryOrder.GROUP_SORT_ALL);
            groupSort.setSortDirection(BySummaryOrder.SORT_DESCENDING_ORDER);*/

            // OR set TopN - shows only N greatest groups
            /*groupSort.setSortType(BySummaryOrder.GROUP_SORT_TOP_N);
            groupSort.setN(2);
            groupSort.setOther(BySummaryOrder.OTHER_RECORDS_GROUP_TO_OTHERS);
            groupSort.setOtherName("NewOtherName");*/

            // OR set BottomN
            groupSort.setSortType( BySummaryOrder.GROUP_SORT_BOTTOM_N );
            groupSort.setN( 2 );
            groupSort.setOther( BySummaryOrder.OTHER_RECORDS_GROUP_TO_OTHERS );
            groupSort.setOtherName( "NewOtherName" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of the sample.
     * @param args no parameters used
     */
    public static void main( String[] args ) {
        new BySummaryOrderSample().initUI();
    }
}
